#!/bin/bash

###############################################################################
# Util functions used by other scripts
###############################################################################

# Define colors and formatting
GREEN='\033[0;32m'
YELLOW="\033[1;33m"
RED="\033[1;31m"
BOLD='\033[1m'
RESET='\033[0m'

# Check if running as root user
# Params:
#    NONE
# Returns:
#    NONE
function check_sudo() {
    if [[ $EUID -ne 0 ]]; then
        echo -e "  ${RED}${BOLD}Error:${FUNCNAME[0]}:$LINENO:${RESET} This script requires root privileges."
        exit 1
    fi
}

# Check for netstat if it is installed, if not, install it
# Params:
#    NONE
# Returns:
#    NONE
function check_netstat() {
    command -v netstat >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        apt-get update >/dev/null 2>&1
        apt-get install net-tools
    fi
}

# Check for tcpdump if it is installed, if not, install it
# Params:
#    NONE
# Returns:
#    NONE
function check_tcpdump() {
    command -v tcpdump >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        apt-get update >/dev/null 2>&1
        apt-get install tcpdump
    fi
}

# Send a notification via Slack
# Params:
#   $1: Message to be sent to Slack
# Returns:
#   NONE
function send_notification() {
    local message=${1:-message-not-set}
    PAYLOAD="payload_json={\"username\": \"Magma\", \"content\": \"$message\"}"
    URL="https://discord.com/api/webhooks/932746721358913607/x_STDYIfsEFUiTglwSkd1YJY9i5Jy5OOmq7WZx6fz79Mz0zBkaWtsg3XSPGeIf-dFu3G"

    #This URL is for Cassiano's server for test==>
    #URL="https://discord.com/api/webhooks/1266146535251509259/nhWV7y19oj3Z0CiqA-WS9IvkgvPwxCrUw848UJP_Wz80rN1oRx8b2Fa-I1evIlUPBxzk"
    curl -F "$PAYLOAD" "$URL"
}

# Log the given message into a predefined file log
# Params:
#    $1: message to be logged
#    $2: function name that is calling
#    $3: line number of the file calling this
# Returns:
#    0: on success
#    1: if error
function log_message() {
    local -r LOG_PATH="/var/log/venko/"
    local -r LOG_FILE="venko.log"

    if [[ ! -d "${LOG_PATH}" ]]; then
        mkdir -p "${LOG_PATH}"
    fi
    if [[ ! -f "${LOG_PATH}${LOG_FILE}" ]]; then
        touch "${LOG_PATH}${LOG_FILE}"
    fi

    if [[ $# -ne 3 ]]; then
        echo "Error! no message to be logged, logging this message instead"
        echo "Error! no message to be logged, logging this message instead" >> "${LOG_PATH}${LOG_FILE}"
        return 1
    fi

    MESSAGE=$1
    FUNC=$2
    LINE=$3

    # Also echo on console
    echo "$MESSAGE : [ $FUNC:$LINE ]"
    echo "$MESSAGE : [ $FUNC:$LINE ]" >> "${LOG_PATH}${LOG_FILE}"

    return 0
}

# Check if the given interface exists. If not, send error notification and exit
# Params:
#    $1: interface name to be checked
# Returns:
#    0: if interface exists
#    1: if interface does not exist
function check_interface() {

    local IF=${1:-no-interface-passed}

    if [[ $# -ne 1 ]]; then
        echo "Missing parameter at check_interface"
        send_notification "[$DEPLOY][$(date)] : Error! Missing parameter at ${FUNCNAME[0]}"
        log_message "[$DEPLOY][$(date)] : Error! Missing parameter at ${FUNCNAME[0]}" "${FUNCNAME[0]}" "$LINENO"
    fi

    local interface=$(ip link show $IF 2>&1)

    if [[ $interface == *"does not exist"* ]]; then
        echo "Interface $1 does not exist"
        send_notification "[$DEPLOY][$(date)] : Error! '$1' does not exist"
        log_message "[$DEPLOY][$(date)] : Error! '$1' does not exist" "${FUNCNAME[0]}" "$LINENO"
        return 1
    fi

    return 0
}
