#!/bin/bash

###############################################################################
# Script used to configure and install all monitoring scripts that will
# be managed by systemd.
###############################################################################

source utils.sh

declare -r FILES_LIST=("check-params.sh" "dump-config.sh" "iptables-config.sh" "iptables.base"
                        "magma-status.sh" "monitor-coredumps.sh" "monitor-memory.sh"
                        "monitor-sctp.sh" "probe-stop.sh" "probe.sh" "utils.sh")
declare -r MAGMA="root"
declare -r PROBE_INSTALL_PATH="/usr/bin/"
declare -r SYSTEMD_SERVICE_FILE="/etc/systemd/system/probe.service"

# Check the files list that must be present to be installed
# Arguments:
#    NONE
# Returns:
#    NONE
function check_files_list() {
    for f in "${FILES_LIST[@]}"; do
        if [[ ! -f "$f" ]]; then
            echo "ERROR: Failed to find file $f"
            exit 1
        fi
    done
}

# Install the necessary files to be managed by the systemd
# Arguments:
#    $1: the timeout for the scripts that capture packets
#    $2: the name of the deployment to be used when sending the notification to Discord
# Returns:
#    NONE
function install_files_list() {
    TIMEOUT=${1:-60}
    DEPLOY=${2:-no-name-set-for-deploy}

    for f in "${FILES_LIST[@]}"; do
        chmod +x "$f"
        cp "$f" $PROBE_INSTALL_PATH || { echo "ERROR: Failed to copy file $f"; exit 1; }
        chown $MAGMA:$MAGMA $PROBE_INSTALL_PATH"$f" || { echo "ERROR: Failed to chown $f"; exit 1; }
    done

    cat <<EOF > $SYSTEMD_SERVICE_FILE
[Unit]
Description=Venko probe script
After=multi-user.target

[Service]
User=root
ExecStart=${PROBE_INSTALL_PATH}probe.sh $TIMEOUT "$DEPLOY"
ExecStop=${PROBE_INSTALL_PATH}probe-stop.sh "$DEPLOY"
Restart=always
RestartSec=30

[Install]
WantedBy=multi-user.target
EOF
}

# Print help message and exit program
# Arguments:
#    NONE
# Returns:
#    NONE
function print_help() {
    echo "Usage: $0 [timeout|<deploy-name>]"
    echo " [OPTION]"
    echo "    timeout: (integer) timeout for tcpdump capture (default: 60)"
    echo "    deploy-name: (string) name of the deploy to be sent in notification messages (default: 'no-name-set-for-deploy')"
    exit 1
}

# Check if the given parameter is a number
# Arguments:
#    NONE
# Returns:
#    0: if is a number
#    1: if it is not a number
function is_a_number() {
    if [[ "$1" =~ ^[0-9]+$ ]]; then
        return 0
    else
        return 1
    fi
}

# Print confirmation for the user before installing scripts
# Arguments:
#    $1: the timeout for the scripts that capture packets
#    $2: the name of the deployment to be used when sending the notification to Discord
# Returns:
#    0: if success
#    exit program: if user does not confirm y or Y.
function confirm_params() {
    TIMEOUT=${1:-60}
    DEPLOY=${2:-no-name-set-for-deploy}

    echo -e "\nConfiguration:"
    echo -e "    Timeout    = $TIMEOUT"
    echo -e "    Deployment = $DEPLOY\n"

    read -p "$(echo -e ${YELLOW}${BOLD}Is the configuration correct? \(y/n\)${RESET} )" RESPONSE
    case $RESPONSE in
        [yY]) ;;
        *) echo -e "\n${RED}${BOLD}Exiting${RESET}! Run the script again with correct parameters\n"; exit 1;;
    esac

    return 0
}

# Check the passed parameters
# If error, prints error message and exit program
# Params:
#    $1: the timeout for the scripts that capture packets
#    $2: the name of the deployment to be used when sending the notification to Discord
# Returns:
#    on error: exit program
#    on success: continue
function check_params() {
    if [[ $# -eq 2 ]]; then
        if ! is_a_number "$1"; then
            echo "Error: First parameter must be a number"
            exit 1
        fi
    elif [[ $# -ne 0 ]]; then
        print_help
    fi
    confirm_params "$@"
}

check_sudo

service probe status | grep "Active: active (running)" >/dev/null
if [[ $? -eq 0 ]]; then
    echo "Stopping 'probe.service'..."
    systemctl stop probe.service 2>/dev/null || true
fi

check_params "$@"
check_files_list
install_files_list "$@"

systemctl daemon-reload
systemctl enable probe.service
systemctl start probe.service
