#!/bin/bash
###############################################################################
# This script is responsible for collecting all the relevant information
# on the current deploy, and collects all the logs necessary for debugging.
###############################################################################
source utils.sh

# Collect the logs on the static paths listed in the function
# It generates the compressed file 'venko_logs_<timestamp>.tar.gz' file
#   in the directory '/var/log/venko/'.
# Arguments:
#     NONE
# Returns:
#     NONE
function __collect_path_logs() {
    local VENKO_FOLDER="/var/log/venko/"
    local INCLUDE_PATHS=("/var/opt/magma/configs/"  \
                        "/etc/magma/"               \
                        "/etc/systemd/system/"      \
                        "/usr/local/bin/mme"        \
                        "/usr/local/bin/ryu"        \
                        "/usr/local/bin/sessiond"   \
                        "/var/core/"                \
                        "/var/log/"                 \
                        )

    OUTPUT_FILE="venko_logs_$(date +%Y-%m-%d-%H%M%S).tar.gz"

    cd "$VENKO_FOLDER" || { echo "Error! Could not find $VENKO_FOLDER"; }

    tar --exclude='venko_logs_*.tar.gz' -czvf "$OUTPUT_FILE" "${INCLUDE_PATHS[@]}"

    echo "Backup created successfully: $(pwd)/$OUTPUT_FILE"
    send_notification "[$DEPLOY][$(date)] : Backup created successfully: $(pwd)/$OUTPUT_FILE"
    __clean_old_logs
    log_message "[$DEPLOY][$(date)] : Backup created successfully: $(pwd)/$OUTPUT_FILE" "${FUNCNAME[0]}" "$LINENO"
}

#Clean logfiles older than 7 days and greater than 100MiBs, for files older than 3 days
function __clean_old_logs () {
    find "/var/log/venko/" -type f -mtime +3 -exec rm -f {} \;
    find "/var/log/venko/" -type f -mtime +1 -size +100M -exec rm -f {} \;
}

# Dump all the relevant statuses of the Magma deployment
# In case of errors, it will send a notification to Discord and also log the error
# Arguments:
#     $1: the name of the deployment to be used when sending the notification to Discord
# Returns:
#     NONE
# Examples:
#     dump_config "foo_bar"
function dump_config() {
    local DEPLOY=${1:-no-deploy-name-set}
    local GET_CONFIGURATION_PATH="/var/log/venko"
    mkdir -p $GET_CONFIGURATION_PATH || {
        send_notification "[$DEPLOY][$(date)] : Could not create directory $GET_CONFIGURATION_PATH, exiting"
        log_message "[$DEPLOY][$(date)] : Could not create directory $GET_CONFIGURATION_PATH, exiting" "${FUNCNAME[0]}" "$LINENO"
    }

    local GET_CONFIGURATION_FILE="$GET_CONFIGURATION_PATH/venko-show-tech-$(date +%Y-%m-%d-%H%M%S).txt"

    if [[ ! -f "$GET_CONFIGURATION_FILE" ]]; then
        touch "$GET_CONFIGURATION_FILE"
    fi

    if [[ ! -f "$GET_CONFIGURATION_FILE" ]]; then
        echo "Error: could not create file $GET_CONFIGURATION_FILE"
        send_notification "[$DEPLOY][$(date)] : Error: could not create file $GET_CONFIGURATION_FILE"
        log_message "[$DEPLOY][$(date)] : Error: could not create file $GET_CONFIGURATION_FILE" "${FUNCNAME[0]}" "$LINENO"
    fi

    echo -e "\nCommand: date\n" >>"$GET_CONFIGURATION_FILE"
    date >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: uptime\n" >>"$GET_CONFIGURATION_FILE"
    uptime >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: uname -a\n" >>"$GET_CONFIGURATION_FILE"
    uname -a >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: top -b -n 1\n" >>"$GET_CONFIGURATION_FILE"
    top -b -n 1 >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: pip3 freeze\n" >>"$GET_CONFIGURATION_FILE"
    pip3 freeze >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: apt list --installed\n" >>"$GET_CONFIGURATION_FILE"
    apt list --installed >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: df -kh\n" >>"$GET_CONFIGURATION_FILE"
    df -kh >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: dpkg -l magma*\n" >>"$GET_CONFIGURATION_FILE"
    dpkg -l magma* >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: dpkg -l *openvswitch*\n" >>"$GET_CONFIGURATION_FILE"
    dpkg -l *openvswitch* >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: ovs-vsctl show\n" >>"$GET_CONFIGURATION_FILE"
    ovs-vsctl show >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: ovs-ofctl show gtp_br0\n" >>"$GET_CONFIGURATION_FILE"
    ovs-ofctl show gtp_br0 >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: ovs-ofctl dump-flows gtp_br0\n" >>"$GET_CONFIGURATION_FILE"
    ovs-ofctl dump-flows gtp_br0 >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: apt show magma\n" >>"$GET_CONFIGURATION_FILE"
    apt show magma >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: service magma@* status\n" >>"$GET_CONFIGURATION_FILE"
    service magma@* status >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: service sctpd status\n" >>"$GET_CONFIGURATION_FILE"
    service sctpd status >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: service openvswitch-switch status\n" >>"$GET_CONFIGURATION_FILE"
    service openvswitch-switch status >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: show_gateway_info.py\n" >>"$GET_CONFIGURATION_FILE"
    show_gateway_info.py >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: checkin_cli.py\n" >>"$GET_CONFIGURATION_FILE"
    checkin_cli.py >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: mobility_cli.py get_subscriber_table\n" >>"$GET_CONFIGURATION_FILE"
    mobility_cli.py get_subscriber_table >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: pipelined_cli.py debug display_flows\n" >>"$GET_CONFIGURATION_FILE"
    pipelined_cli.py debug display_flows >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: enodebd_cli.py get_all_status\n" >>"$GET_CONFIGURATION_FILE"
    enodebd_cli.py get_all_status >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: ip addr\n" >>"$GET_CONFIGURATION_FILE"
    ip addr >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: ping google.com -I eth0 -c 5\n" >>"$GET_CONFIGURATION_FILE"
    ping google.com -I eth0 -c 5 >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: ip route\n" >>"$GET_CONFIGURATION_FILE"
    ip route >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: iptables -L\n" >>"$GET_CONFIGURATION_FILE"
    iptables -L >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: iptables -L -v -n\n" >>"$GET_CONFIGURATION_FILE"
    iptables -L -v -n >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: iptables -t nat -L -v -n\n" >>"$GET_CONFIGURATION_FILE"
    iptables -t nat -L -v -n >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: iptables -t mangle -L -v -n\n" >>"$GET_CONFIGURATION_FILE"
    iptables -t mangle -L -v -n >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: iptables -t raw -L -v -n\n" >>"$GET_CONFIGURATION_FILE"
    iptables -t raw -L -v -n >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: iptables -t security -L -v -n\n" >>"$GET_CONFIGURATION_FILE"
    iptables -t security -L -v -n >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: netstat -i\n" >>"$GET_CONFIGURATION_FILE"
    netstat -i >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: cat /proc/net/sctp/snmp\n" >>"$GET_CONFIGURATION_FILE"
    cat /proc/net/sctp/snmp >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: cat /proc/net/sctp/remaddr\n" >>"$GET_CONFIGURATION_FILE"
    cat /proc/net/sctp/remaddr >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: cat /proc/net/sctp/eps\n" >>"$GET_CONFIGURATION_FILE"
    cat /proc/net/sctp/eps >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    echo -e "\nCommand: cat /proc/net/sctp/assocs\n" >>"$GET_CONFIGURATION_FILE"
    cat /proc/net/sctp/assocs >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    ip_enb=$(cat /proc/net/sctp/remaddr | awk 'FNR==2{print $1}')

    echo -e "\nCommand: ping -c 10 -i 0.5 $ip_enb\n" >>"$GET_CONFIGURATION_FILE"
    ping -c 10 -i 0.5 $ip_enb >>"$GET_CONFIGURATION_FILE"
    echo -e "\n" >>"$GET_CONFIGURATION_FILE"

    services=("magma@smsd" "magma@mobilityd" "magma@td-agent-bit" "magma@policydb" "magma@redis" "magma@envoy_controller" "magma@enodebd" "magma@eventd" "magma@sessiond" "magma@magmad" "magma@subscriberdb" "magma@control_proxy" "magma@pipelined" "magma@dnsd" "magma@ctraced" "magma@directoryd" "magma@health" "magma@state" "magma@mme")
    for service in "${services[@]}"
    do
        echo -e "\nCommand: journalctl --since \"30 min ago\" -u \"$service\"\n" >>"$GET_CONFIGURATION_FILE"
        journalctl --since "30 min ago" -u $service >>"$GET_CONFIGURATION_FILE"
        echo -e "\n" >>"$GET_CONFIGURATION_FILE"
    done

    services=("syslog" "mme.log" "openvswitch/ovs-vswitchd.log")
    for service in "${services[@]}"
    do
        echo -e "\nCommand: tail -100 /var/log/$service\n" >>"$GET_CONFIGURATION_FILE"
        tail -100 /var/log/$service >>"$GET_CONFIGURATION_FILE"
        echo -e "\n" >>"$GET_CONFIGURATION_FILE"
    done

    __collect_path_logs
}
