#!/usr/bin/env python3
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
#  Distributed as Tibit-Customer confidential.                             #
#                                                                          #
#--------------------------------------------------------------------------#
""" Get firmware information for an ONU.

This MCMS REST API example script retrieves firmware information for the
specified ONU, including the current active firmware version, firmware bank
configuration, and a list of firmware images available for the ONU. The list
of available firmware images is based on the vendor name as reported by
the ONU.

Example:

  ./get_onu_firmware.py --url https://10.2.10.29/api --user <email> --password <password> --onu TBITc84c0083

usage: get_onu_firmware.py [-d DATABASE] [-f FILTER] [-h] [-l URL] --onu ONU
                           [-p PASSWORD] [-u USER] [-v]

optional arguments:
  -d DATABASE, --db DATABASE
                        Name of the database. (default: Default)
  -f FILTER, --filter FILTER
                        Filter by ONU state (e.g., Dregistered, Disabled,
                        Disallowed-Admin, Disallowed-Error, Dying-Gasp,
                        Registered, Unprovisioned, Unspecified) (default:
                        None)
  -h, --help            Show this help message and exit.
  -l URL, --url URL     URL of the MCMS API server (e.g.,
                        https://10.2.10.29/api). (default:
                        https://10.2.10.29/api)
  --onu ONU             ONU Serial Number (e.g., TBITc84c00df) (default: None)
  -p PASSWORD, --password PASSWORD
                        User password to authenticate with. (default: tibit)
  -u USER, --user USER  User email to authenticate with. (default:
                        tibit@tibitcom.com)
  -v, --verbose         Verbose output. (default: False)


"""

import argparse
import sys
from api_client import ApiClient


def main():
    """ Entry point for the script. """
    parser = argparse.ArgumentParser(add_help=False, formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument("-d", "--db", action="store", dest="database", default="Default", required=False, help="Name of the database.")
    parser.add_argument("-f", "--filter", action="store", dest="filter", default=None, required=False, help="Filter by ONU state (e.g., Dregistered, Disabled, Disallowed-Admin, Disallowed-Error, Dying-Gasp, Registered, Unprovisioned, Unspecified)")
    parser.add_argument("-h", "--help", action="help", default=argparse.SUPPRESS, help="Show this help message and exit.")
    parser.add_argument("-l", "--url", action="store", dest="url", default="https://10.2.10.29/api", required=False, help="URL of the MCMS API server (e.g., https://10.2.10.29/api).")
    parser.add_argument("--onu", action="store", dest="onu", default=None, required=True, help="ONU Serial Number (e.g., TBITc84c00df)")
    parser.add_argument("-p", "--password", action="store", dest="password", default="tibit", required=False, help="User password to authenticate with.")
    parser.add_argument("-u", "--user", action="store", dest="user", default="tibit@tibitcom.com", required=False, help="User email to authenticate with.")
    parser.add_argument("-v", "--verbose", action="store_true", dest="verbose", default=False, required=False, help="Verbose output.")
    parser.parse_args()
    args = parser.parse_args()


    # Instantiate an API Client Connection
    api_client = ApiClient(args.url, args.verbose)

    # Login to the web server
    api_client.login(args.user, args.password)

    # Select the database to use for this session
    api_client.select_database(args.database)

    # Get the ONU-CFG for this ONU
    status, onu_cfg = api_client.request("GET", f"/v1/onus/configs/{args.onu}/")
    if status != 200 or not onu_cfg:
        print(f"ERROR: Failed to read state for ONU {args.onu}.")
        sys.exit(1)

    # Parse the Firmware configuration information
    onu_cfg_fw_bank_ptr = onu_cfg["ONU"]["FW Bank Ptr"]
    onu_cfg_fw_bank_versions = []
    if len(onu_cfg["ONU"]["FW Bank Versions"]) > 0 and onu_cfg["ONU"]["FW Bank Versions"][0]:
        onu_cfg_fw_bank_versions.append(onu_cfg["ONU"]["FW Bank Versions"][0])
    else:
        onu_cfg_fw_bank_versions.append("None")
    if len(onu_cfg["ONU"]["FW Bank Versions"]) > 1 and onu_cfg["ONU"]["FW Bank Versions"][1]:
        onu_cfg_fw_bank_versions.append(onu_cfg["ONU"]["FW Bank Versions"][1])
    else:
        onu_cfg_fw_bank_versions.append("None")
    onu_cfg_fw_bank_files = []
    if len(onu_cfg["ONU"]["FW Bank Files"]) > 0 and onu_cfg["ONU"]["FW Bank Files"][0]:
        onu_cfg_fw_bank_files.append(onu_cfg["ONU"]["FW Bank Files"][0])
    else:
        onu_cfg_fw_bank_files.append("None")
    if len(onu_cfg["ONU"]["FW Bank Files"]) > 1 and onu_cfg["ONU"]["FW Bank Files"][1]:
        onu_cfg_fw_bank_files.append(onu_cfg["ONU"]["FW Bank Files"][1])
    else:
        onu_cfg_fw_bank_files.append("None")

    # Get the ONU-STATE for this ONU
    status, onu_state = api_client.request("GET", f"/v1/onus/states/{args.onu}/")
    if status != 200 or not onu_state:
        print(f"INFO: Preprovisioned ONU {args.onu} has no state.")

    # Parse the Firmware Information from state
    if onu_state:
        onu_state_fw_bank_ptr = onu_state["ONU"]["FW Bank Ptr"]
        onu_state_fw_bank_versions = []
        if len(onu_state["ONU"]["FW Bank Versions"]) > 0 and onu_state["ONU"]["FW Bank Versions"][0]:
            onu_state_fw_bank_versions.append(onu_state["ONU"]["FW Bank Versions"][0])
        else:
            onu_state_fw_bank_versions.append("None")
        if len(onu_state["ONU"]["FW Bank Versions"]) > 1 and onu_state["ONU"]["FW Bank Versions"][1]:
            onu_state_fw_bank_versions.append(onu_state["ONU"]["FW Bank Versions"][1])
        else:
            onu_state_fw_bank_versions.append("None")

    # Get the vendor name for this ONU
    vendor_name = None
    if onu_state:
        vendor_name = onu_state['ONU']['Vendor']
        if vendor_name.upper() == "TBIT" or vendor_name.upper().startswith("TIBIT"):
            vendor_name = "TIBITCOM"

    # Get a list of available firmware for this ONU
    available_firmware = []
    if onu_state and vendor_name:
        status, firmware_files = api_client.request("GET", f"/v1/files/onu-firmware/?query=metadata.Compatible%20Manufacturer={vendor_name}")
        if status == 200 and firmware_files:
            for file in firmware_files:
                available_firmware.append({
                    "Vendor": file['metadata']['Compatible Manufacturer'],
                    "Model": file['metadata']['Compatible Model'][0],
                    "Version": file['metadata']['Version'],
                    "Filename": file['filename']
                })

    # Display the firmare information for the ONU
    print(f"\nFirmware Information for ONU {args.onu}:")
    print()
    # Show the current version information from state
    if onu_state:
        print("Current Versions:")
        print(f"  Bank 0:  {onu_state_fw_bank_versions[0]} {'(active)' if onu_state_fw_bank_ptr == 0 else ''}")
        print(f"  Bank 1:  {onu_state_fw_bank_versions[1]} {'(active)' if onu_state_fw_bank_ptr == 1 else ''}")
    else:
        print("Current Version: None")
    print()

    # Show the configured firmware version information
    if onu_cfg_fw_bank_ptr != 65535:
        print(f"Configuration:")
        print("           {:<16} {}".format("Version", "Filename"))
        print("  Bank 0:  {:<16} {} {}".format(
            onu_cfg_fw_bank_versions[0], onu_cfg_fw_bank_files[0], '(selected)' if onu_cfg_fw_bank_ptr == 0 else ''))
        print("  Bank 1:  {:<16} {} {}".format(
            onu_cfg_fw_bank_versions[1], onu_cfg_fw_bank_files[1], '(selected)' if onu_cfg_fw_bank_ptr == 1 else ''))
    else:
        print(f"Configuration: Upgrade Disabled")
    print()

    # Last upgrade status from state
    if onu_state and onu_state["ONU"]["FW Upgrade Status"]:
        print("Last Firmware Upgrade Status:")
        print(f"  Status:     {onu_state['ONU']['FW Upgrade Status']['Status']}")
        print(f"  Bank:       {onu_state['ONU']['FW Upgrade Status']['Bank']}")
        print(f"  File:       {onu_state['ONU']['FW Upgrade Status']['File']}")
        print(f"  Progress:   {onu_state['ONU']['FW Upgrade Status']['Progress']} %")
    else:
        print("Last Firmware Upgrade Status: None")
    print()

    # List of available firmware available for this ONU
    print(f"List of firmware available for this ONU ({vendor_name}):")
    print("{:<12} {:<18} {:<16} {}".format("Vendor", "Model", "Version", "Filename"))
    for version in available_firmware:
        print("{:<12} {:<18} {:<16} {}".format(
            version["Vendor"],
            version["Model"],
            version["Version"],
            version["Filename"]))


if __name__ == '__main__':
    main()
