"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import datetime
import hashlib
import json

from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.decorators import permission_required
from django.contrib.auth.models import User
from django.core import serializers
from django.http import JsonResponse
from django.utils.decorators import method_decorator
from drf_spectacular.types import OpenApiTypes
from rest_framework.fields import EmailField, CharField
from rest_framework.generics import GenericAPIView
from drf_spectacular.utils import extend_schema, OpenApiParameter
from rest_framework import status

import DBdjango
from database_manager import database_manager
from utils.schema_helpers import ResponseExample
from utils.serializers import RequestSerializer
from utils.tools import PonManagerApiResponse


# ==================================================
# ============ Authenticate User View ==============
# ==================================================
class Authenticate(GenericAPIView):
    queryset = ''

    @extend_schema(
        operation_id="authenticate_user",
        request=RequestSerializer(name="AuthenticateUserData", data_fields={
            "email": EmailField(),
            "password": CharField()
        }),
        responses={
            200: ResponseExample(200),
            401: ResponseExample(401),
            500: ResponseExample(500),
        },
        tags=['user', 'authenticate', 'login']
    )
    def post(self, request, version):
        """Authenticate and login the user"""
        DBdjango.views.purge_expired_sessions(request)
        email = request.data['email'].lower()
        password = hashlib.md5(request.data['password'].encode('utf-8')).hexdigest()

        # This is responsible for actual authentication. If it succeeds, it returns an object of user data. If it fails, it returns nothing
        user = authenticate(username=email, password=password)

        # If user is not null, that means it returned user data, which means user authentication was successful
        if user is not None:
            user = User.objects.get(username=email)  # Getting user object based on primary key Username( which is email)
            user.last_login = datetime.datetime.now(datetime.timezone.utc)
            user.save()

            # Set user session timeout
            # Database value is in minutes. However, the set_expiry method takes seconds.
            timeout_sec = database_manager.get_user_session_expiry(email)
            timeout_min = timeout_sec * 60
            request.session.set_expiry(timeout_min)

            # Get and save the user selected database in the users session
            try:
                active_database = database_manager.get_users_selected_database(user_email=email)
            except KeyError:
                print("Users selected database was not found. Using 'Default'")
                active_database = "Default"
                database_manager.set_users_selected_database(user_email=email, database_id=active_database)

            # Update private endpoints database reference
            DBdjango.views.clients[email] = active_database

            # Log in the user
            login(request, user)

            # User data to be used in view context
            data = {
                'auth': 'True',
                'fName': user.first_name,
                'lName': user.last_name,
                'email': user.email,
                'lastLogin': user.last_login,
                'dateJoined': user.date_joined,
                'roles': list(map(lambda u: u['fields']['name'], json.loads(serializers.serialize('json', user.groups.all())))),
                'permissions': list(User.get_all_permissions(user)),
                'active_database': active_database
            }

            response = JsonResponse(status=status.HTTP_200_OK, data=data)
        else:
            response = JsonResponse(status=status.HTTP_401_UNAUTHORIZED, data={'auth': 'False'})

        return response


# ==================================================
# =============== Logout User View =================
# ==================================================
class Logout(GenericAPIView):
    queryset = ''

    @extend_schema(
        operation_id="logout_user",
        responses={
            204: ResponseExample(204)
        },
        tags=['user', 'logout']
    )
    def get(self, request, version):
        """ Logout the user """
        # Logout always return successfully regardless if user is logged in/exists
        logout(request)
        response = PonManagerApiResponse(status=status.HTTP_204_NO_CONTENT)
        response.delete_cookie('sessionexpire')
        response.delete_cookie('csrftoken')
        response.delete_cookie('sessionid')
        response.delete_cookie('__Host-sessionexpire')
        response.delete_cookie('__Host-csrftoken')
        response.delete_cookie('__Host-sessionid')

        return response


# ==================================================
# ================ User Logs View ==================
# ==================================================
class Logs(GenericAPIView):
    queryset = ''

    @extend_schema(
        operation_id="get_user_logs",
        parameters=[
            OpenApiParameter(name="start-time", description="UTC timestamp to begin getting logs at",
                             type=OpenApiTypes.DATETIME, required=True)
        ],
        responses={
            200: ResponseExample(200),
            400: ResponseExample(400),
            403: ResponseExample(403),
            500: ResponseExample(500)
        },
        tags=['user', 'logs', 'get']
    )
    @method_decorator(permission_required('accounts.can_read_accounts_admin', raise_exception=True))
    def get(self, request, version):
        """ Get all logs """
        start_time = request.GET.get('start-time', None)

        # Return missing parameter response if start time is undefined
        if start_time is None:
            response = PonManagerApiResponse(status=status.HTTP_400_BAD_REQUEST, data="Parameter 'start-time' is required")
        else:
            collection = database_manager.user_database.get_collection("SYSLOG-ACTIONS")
            logs = collection.find({"Time": {"$gt": start_time}}, no_cursor_timeout=True)
            response = PonManagerApiResponse(status=status.HTTP_200_OK, data=list(logs))

        return response
