"""
#--------------------------------------------------------------------------#
# Copyright (C) 2022 by Tibit Communications, Inc.                         #
# All rights reserved.                                                     #
#                                                                          #
#    _______ ____  _ ______                                                #
#   /_  __(_) __ )(_)_  __/                                                #
#    / / / / __  / / / /                                                   #
#   / / / / /_/ / / / /                                                    #
#  /_/ /_/_____/_/ /_/                                                     #
#                                                                          #
# Distributed as Tibit-Customer confidential.                              #
#                                                                          #
#--------------------------------------------------------------------------#
"""

import sys
import traceback

from django.contrib.auth.decorators import permission_required
from django.contrib.auth.mixins import LoginRequiredMixin
from django.utils.decorators import method_decorator
from rest_framework.views import APIView
from drf_spectacular.utils import extend_schema, OpenApiParameter
from drf_spectacular.types import OpenApiTypes
from rest_framework.response import Response
from rest_framework import status

from database_manager import database_manager
from utils.schema_helpers import ResponseExample


# ==================================================
# ========= One ONU Upgrade Statuses View ==========
# ==================================================
class UpgradeStatus(LoginRequiredMixin, APIView):
    raise_exception = True
    queryset = ''

    @extend_schema(
        exclude=True,
        operation_id="get_onu_upgrade",
        parameters=[
            OpenApiParameter(name="onu-id", description="ID of the ONU to get the firmware upgrade status for", type=OpenApiTypes.STR)
        ],
        responses={
            200: ResponseExample(200),
            403: ResponseExample(403),
            500: ResponseExample(500),
        },
        tags=['onu', 'upgrade', 'firmware', 'get']
    )
    @method_decorator(permission_required('network.can_read_network_onus', raise_exception=True))
    def get(self, request, onu_id):
        """Get the firmware upgrade status for the specified ONU"""

        try:
            database = database_manager.get_database(request.session.get('database'))
            collection = database.get_collection("ONU-STATE")
            res_data = collection.find_one({'_id': onu_id}, {"ONU.FW Upgrade Status": 1})
            response = Response(status=status.HTTP_200_OK, data=res_data)
        except Exception as e:
            traceback.print_exc(file=sys.stdout)
            response = Response(status=status.HTTP_500_INTERNAL_SERVER_ERROR)

        return response
